; Superclass.asm - demonstrates window superclassing
;
; Window subclassing is a useful technique for extending the abilities of any
; child window control. However, it does have one serious limitation - imagine
; you've got a number of controls you'd like to subclass in the same way.
; Subclassing one or two isn't too much trouble but any more and it becomes a
; serious pain, not to mention that it makes maintenance a nightmare.
;
; Window superclassing to the rescue.
;
; Superclassing is a way of taking an existing window class and extending
; it, like subclassing, except that we can then use the new class (which is
; like a subclassed window) when we create any number of child window
; controls. This makes our job a lot easier.
;
; Superclassing involves several steps:
;
;   1. Use GetClassInfoEx to populate a WNDCLASSEX structure with the
;      details of the base window class.
;   2. Fill in the missing details for the new class
;   3. Register the new class
;
; In this example, I'll superclass an edit control to make one which only
; accepts hexadecimal numbers (like in the subclassing example) and then
; make four child window controls out of our new one. You will note how
; much easier this is than subclassing each one.
;

%define _WINMESSAGES_
%define _WINVKEYS_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

ddglobal _gbl_hInstance

procglobal WinMain, hInstance, hPrevInstance, lpszCmdLine, nCmdShow
	ddlocal		_hwnd
	struclocal	_wndclass, WNDCLASSEX, _msg, MSG
	endlocals
	WinMainPrologue
	mov	eax, .hInstance
	mov	._gbl_hInstance, eax
	mov	esi, ._wndclass
	mov	edi, ._msg
	mov	[esi + WNDCLASSEX.cbSize], dword WNDCLASSEX_size
	mov	[esi + WNDCLASSEX.style], dword CS_HREDRAW | CS_VREDRAW
	mov	[esi + WNDCLASSEX.lpfnWndProc], dword _WndProc
	mov	[esi + WNDCLASSEX.cbClsExtra], dword 0
	mov	[esi + WNDCLASSEX.cbWndExtra], dword 0
	mov	eax, .hInstance
	mov	[esi + WNDCLASSEX.hInstance], eax
	sc LoadIcon, NULL, IDI_APPLICATION
	mov	[esi + WNDCLASSEX.hIcon], eax
	sc LoadCursor, NULL, IDC_ARROW
	mov	[esi + WNDCLASSEX.hCursor], eax
	sc GetStockObject, WHITE_BRUSH
	mov	[esi + WNDCLASSEX.hbrBackground], eax
	mov	[esi + WNDCLASSEX.lpszMenuName], dword NULL
	TEXTlocal szClassName, 'MyClass',0
	mov	[esi + WNDCLASSEX.lpszClassName], dword .szClassName
	sc RegisterClassEx, esi
	cmp	eax, TRUE
	je	near _WinMain_Fail
	TEXTlocal szWndCaption, 'Superclassing an edit control',0
	sc CreateWindowEx, 0, .szClassName, .szWndCaption, WS_OVERLAPPEDWINDOW | WS_VISIBLE, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, .hInstance, NULL
	mov	._hwnd, eax
	sc ShowWindow, ._hwnd, .nCmdShow
	sc UpdateWindow, ._hwnd
_WinMain_Loop:
	sc GetMessage, ._msg, NULL, 0, 0
	cmp	eax, TRUE
	jne	_WinMain_Loop_End
	sc TranslateMessage, ._msg
	sc DispatchMessage, ._msg
	jmp	_WinMain_Loop
_WinMain_Loop_End:
	mov	eax, [edi + MSG.wParam]
	jmp	_WinMain_End
_WinMain_Fail:
	TEXTlocal szErrorMsg, 'Failed to register window class!',0
	sc MessageBox, NULL, .szErrorMsg, .szWndCaption, MB_ICONERROR
_WinMain_End:
	WinMainEpilogue
endproc
;
;-----------------------------------------------------------------------
;
proc _WndProc, hwnd, message, wParam, lParam
	buffglobal	_hwndArray, 4*4
	struclocal	_superwndclass, WNDCLASSEX
	ddglobal	_hwndProc
	endlocals
	;
	CallbackPrologue
	;
	switch .message
		case WM_CREATE
			;
			; To superclass a class we need to get the information about an
			; existing class via GetClassInfoEx which takes three parameters:
			;
			;   1. Handle of application that created the class. Use NULL
			;      for classes predefined by Windows
			;   2. Pointer to the class name
			;   3. Pointer to a WNDCLASSEX structure that will be populated
			;      with the class' details
			;
			mov	esi, ._superwndclass
			mov	[esi + WNDCLASSEX.cbSize], dword WNDCLASSEX_size
			sc GetClassInfoEx, NULL, _szClassNameEdit, esi
			;
			; Secondly, we need to fill in the missing details of our new
			; class - GetClassInfoEx does not return the lpszMenuName,
			; lpszClassName or hInstance members of the structure and we
			; need to fill these in ourself.
			;
			mov	[esi + WNDCLASSEX.lpszMenuName], dword NULL
			TEXTlocal _szSuperClassName, 'MySuperClass',0
			mov	[esi + WNDCLASSEX.lpszClassName], dword ._szSuperClassName
			mov	eax, ._gbl_hInstance
			mov	[esi + WNDCLASSEX.hInstance], eax
			;
			; We also need to tell it to use our window procedure, but
			; before we do, we'll save the old (ie predefined by Windows)
			; window procedure's address
			;
			mov	eax, [esi + WNDCLASSEX.lpfnWndProc]
			mov	._hwndProc, eax
			mov	[esi + WNDCLASSEX.lpfnWndProc], dword _SuperclassProc
			;
			; And before we use the class we need to register it
			;
			sc RegisterClassEx, esi
			;
			; Finally, create four edit controls with our new window class
			;
			mov	esi, ._hwndArray
			sc CreateWindowEx, 0, ._szSuperClassName, 0, WS_CHILDWINDOW | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL | ES_MULTILINE | ES_NOHIDESEL | ES_WANTRETURN, 0, 0, 0, 0, .hwnd, 1, ._gbl_hInstance, NULL
			mov	[esi], eax
			sc CreateWindowEx, 0, ._szSuperClassName, 0, WS_CHILDWINDOW | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL | ES_MULTILINE | ES_NOHIDESEL | ES_WANTRETURN, 0, 0, 0, 0, .hwnd, 2, ._gbl_hInstance, NULL
			mov	[esi + 4], eax
			sc CreateWindowEx, 0, ._szSuperClassName, 0, WS_CHILDWINDOW | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL | ES_MULTILINE | ES_NOHIDESEL | ES_WANTRETURN, 0, 0, 0, 0, .hwnd, 3, ._gbl_hInstance, NULL
			mov	[esi + 8], eax
			sc CreateWindowEx, 0, ._szSuperClassName, 0, WS_CHILDWINDOW | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL | ES_MULTILINE | ES_NOHIDESEL | ES_WANTRETURN, 0, 0, 0, 0, .hwnd, 4, ._gbl_hInstance, NULL
			mov	[esi + 12], eax
			xor	eax,eax
			break
		case WM_SIZE
			;
			; resize the four edit controls
			;
			mov	eax, .lParam
			movsx	esi, ax
			shr	eax, 16
			movsx	edi, ax
			;
			; divide width and height by two
			;
			shr	esi, 1
			shr	edi, 1
			;
			; ebx -> array of window handles
			;
			mov	ebx, ._hwndArray
			sc MoveWindow, [ebx], 0, 0, esi, edi, TRUE
			sc MoveWindow, [ebx + 4], esi, 0, esi, edi, TRUE
			sc MoveWindow, [ebx + 8], 0, edi, esi, edi, TRUE
			sc MoveWindow, [ebx + 12], esi, edi, esi, edi, TRUE
			xor	eax,eax
			break
		case WM_DESTROY
			sc PostQuitMessage, 0
			xor	eax,eax
			break
		default
			sc DefWindowProc, .hwnd, .message, .wParam, .lParam
	switchend
	;
	CallbackEpilogue
endproc
;
;-----------------------------------------------------------------------
;
; The superclass window procedure - identical to the subclassed
; procedure that was used in the subclassing example
;
proc _SuperclassProc, hwnd, message, wParam, lParam
	;
	CallbackPrologue
	;
	switch .message
		case WM_CHAR
			mov	eax, .wParam
			xor	ecx, ecx
			if al, e, VK_BACK
				mov	ecx, 1
			else
				ifand al, ge, '0', al, le, '9'
					mov	ecx, 1
				else
					ifand al, ge, 'A', al, le, 'F'
						mov	ecx, 1
					else
						ifand al, ge, 'a', al, le, 'f'
							mov	ecx, 1
						endif
					endif
				endif
			endif
			if ecx, e, 0
				xor	eax, eax
			else
				sc CallWindowProc, ._hwndProc, .hwnd, WM_CHAR, eax, .lParam
			endif
			break
		default
			sc CallWindowProc, ._hwndProc, .hwnd, .message, .wParam, .lParam
	switchend
	;
	CallbackEpilogue
endproc
;
;-----------------------------------------------------------------------
;
[section .bss]

[section .data]
